<?php

namespace App\Http\Controllers;

use App\BusinessLocation;
use App\Category;
use App\PurchaseLine;
use App\TaxRate;
use App\Transaction;
use App\TransactionComment;
use App\Utils\TransactionUtil;
use App\Utils\Util;
use App\VariationLocationDetails;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Yajra\DataTables\Facades\DataTables;

class PurchaseRequisitionController extends Controller
{
    protected $commonUtil;

    protected $transactionUtil;

    /**
     * Map of PR statuses used across UI and logic.
     * @var array<string,array{label:string,class:string}>
     */
    protected $purchaseRequisitionStatuses;

    /**
     * Memoized allowed statuses for the current user (per request).
     * @var array<string,string>
     */
    private $allowedStatusCache = [];

    /**
     * Constructor
     *
     * @param  Util  $commonUtil
     * @return void
     */
    public function __construct(Util $commonUtil, TransactionUtil $transactionUtil)
    {
        $this->commonUtil = $commonUtil;
        $this->transactionUtil = $transactionUtil;

        $this->purchaseRequisitionStatuses = [
            'requested' => [
                'label' => 'Requested',
                'class' => 'bg-light-blue',
            ],
            'quoted' => [
                'label' => 'Quoted',
                'class' => 'bg-blue',
            ],
            'for_approval' => [
                'label' => 'For Approval',
                'class' => 'bg-orange',
            ],
            'for_product_creation' => [
                'label' => 'For Product Creation',
                'class' => 'bg-yellow',
            ],
            'request_transfer' => [
                'label' => 'Request Transfer',
                'class' => 'bg-light-blue',
            ],
            'transfer_approved' => [
                'label' => 'Transfer Approved',
                'class' => 'bg-green',
            ],
            'approved' => [
                'label' => 'Approved',
                'class' => 'bg-green',
            ],
            'rejected' => [
                'label' => 'Rejected',
                'class' => 'bg-red',
            ],
        ];
    }

    /**
     * Check if current user can set a PR to the given status.
     */
    private function canSetPrStatus(string $status): bool
    {
        $user = auth()->user();
        if (! $user) {
            return false;
        }
        // Specific per-status permission
        $perm = 'purchase_requisition.status.' . $status;
        if ($user->can($perm)) {
            return true;
        }
        // Backward-compat: allow legacy approve permission for approval-like statuses
        if (in_array($status, ['approved', 'transfer_approved'], true) && $user->can('purchase_requisition.approve')) {
            return true;
        }
        return false;
    }

    /**
     * Return associative array of allowed statuses => labels for the current user.
     */
    private function getAllowedPrStatuses(): array
    {
        if (!empty($this->allowedStatusCache)) {
            return $this->allowedStatusCache;
        }
        $allowed = [];
        foreach ($this->purchaseRequisitionStatuses as $key => $meta) {
            if ($this->canSetPrStatus($key)) {
                $allowed[$key] = $meta['label'];
            }
        }
        return $this->allowedStatusCache = $allowed;
    }

    /**
     * Find a sensible default allowed status for the current user.
     * Prefer 'requested' if allowed, else the first allowed in configured order.
     */
    private function getDefaultAllowedPrStatus(): ?string
    {
        if ($this->canSetPrStatus('requested')) {
            return 'requested';
        }
        foreach (array_keys($this->purchaseRequisitionStatuses) as $key) {
            if ($this->canSetPrStatus($key)) {
                return $key;
            }
        }
        return null;
    }

    /**
     * Allowed priority levels for PRs.
     *
     * @return array<string,string>
     */
    private function priorityLevels(): array
    {
        return [
            '' => __('lang_v1.priority_blank'),
            'sure_order' => __('lang_v1.priority_sure_order'),
            'rush' => __('lang_v1.priority_rush'),
        ];
    }

    /**
     * Sanitize incoming priority.
     */
    private function sanitizePriorityLevel($value): ?string
    {
        if ($value === '' || $value === null) {
            return null;
        }
        $allowed = array_keys($this->priorityLevels());
        return in_array($value, $allowed, true) ? $value : null;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if (! auth()->user()->can('purchase_requisition.view_all') && ! auth()->user()->can('purchase_requisition.view_own')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        if (request()->ajax()) {
            $purchase_requisitions = Transaction::join(
                        'business_locations AS BS',
                        'transactions.location_id',
                        '=',
                        'BS.id'
                    )
                    ->join('users as u', 'transactions.created_by', '=', 'u.id')
                    ->leftJoin('users as au', 'transactions.approved_by', '=', 'au.id')
                    ->leftJoin('users as uu', 'transactions.updated_by', '=', 'uu.id')
                    ->where('transactions.business_id', $business_id)
                    ->where('transactions.type', 'purchase_requisition')
                    ->select(
                        'transactions.id',
                        'transactions.delivery_date',
                        'transactions.ref_no',
                        'transactions.status',
                        'transactions.approval_status',
                        'transactions.priority_level',
                        'transactions.additional_notes',
                        'BS.name as location_name',
                        'transactions.transaction_date',
                        DB::raw("CONCAT(COALESCE(u.surname, ''),' ',COALESCE(u.first_name, ''),' ',COALESCE(u.last_name,'')) as added_by"),
                        DB::raw("CONCAT(COALESCE(au.surname, ''),' ',COALESCE(au.first_name, ''),' ',COALESCE(au.last_name,'')) as approved_by_name"),
                        DB::raw("CONCAT(COALESCE(uu.surname, ''),' ',COALESCE(uu.first_name, ''),' ',COALESCE(uu.last_name,'')) as updated_by_name")
                    )
                    ->groupBy('transactions.id');

            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $purchase_requisitions->whereIn('transactions.location_id', $permitted_locations);
            }

            if (! empty(request()->location_id)) {
                $purchase_requisitions->where('transactions.location_id', request()->location_id);
            }

            if (! empty(request()->status)) {
                $purchase_requisitions->where('transactions.status', request()->status);
            }

            if (! empty(request()->start_date) && ! empty(request()->end_date)) {
                $start = request()->start_date;
                $end = request()->end_date;
                $purchase_requisitions->whereDate('transactions.transaction_date', '>=', $start)
                            ->whereDate('transactions.transaction_date', '<=', $end);
            }

            if (! empty(request()->required_by_start) && ! empty(request()->required_by_end)) {
                $start = request()->required_by_start;
                $end = request()->required_by_end;
                $purchase_requisitions->whereDate('transactions.delivery_date', '>=', $start)
                            ->whereDate('transactions.delivery_date', '<=', $end);
            }
            $priority_levels = $this->priorityLevels();

            if (! auth()->user()->can('purchase_requisition.view_all') && auth()->user()->can('purchase_requisition.view_own')) {
                $purchase_requisitions->where('transactions.created_by', request()->session()->get('user.id'));
            }

            if (! empty(request()->from_dashboard)) {
                $purchase_requisitions->where('transactions.status', '!=', 'completed');
            }

            return Datatables::of($purchase_requisitions)
                ->filter(function ($query) {
                    $search = request()->input('search.value') ?? request()->input('search')['value'] ?? '';
                    $search = trim((string)$search);
                    if ($search === '') {
                        return;
                    }
                    $like = '%' . str_replace(['%','_'], ['\\%','\\_'], $search) . '%';
                    $query->where(function ($q) use ($like) {
                        $q->where('transactions.ref_no', 'like', $like)
                          ->orWhere('transactions.additional_notes', 'like', $like)
                          ->orWhereExists(function ($sq) use ($like) {
                              $sq->from('purchase_lines as plf')
                                  ->join('products as pf', 'plf.product_id', '=', 'pf.id')
                                  ->whereColumn('plf.transaction_id', 'transactions.id')
                                  ->where(function ($qq) use ($like) {
                                      $qq->where('pf.name', 'like', $like)
                                         ->orWhere('pf.sku', 'like', $like)
                                         ->orWhere('plf.quantity', 'like', $like);
                                  });
                          });
                    });
                })
                ->addColumn('action', function ($row) {
                    $html = '<div class="btn-group">
                            <button type="button" class="tw-dw-btn tw-dw-btn-xs tw-dw-btn-outline  tw-dw-btn-info tw-w-max  dropdown-toggle" 
                                data-toggle="dropdown" aria-expanded="false">'.
                                __('messages.actions').
                                '<span class="caret"></span><span class="sr-only">Toggle Dropdown
                                </span>
                            </button>
                            <ul class="dropdown-menu dropdown-menu-left" role="menu">';
                    $html .= '<li><a href="#" data-href="'.action([\App\Http\Controllers\PurchaseRequisitionController::class, 'show'], [$row->id]).'" class="btn-modal" data-container=".view_modal"><i class="fas fa-eye" aria-hidden="true"></i>'.__('messages.view').'</a></li>';
                    if (auth()->user()->can('purchase_requisition.create') && $row->status !== 'rejected') {
                        $html .= '<li><a href="'.action([\App\Http\Controllers\PurchaseRequisitionController::class, 'edit'], [$row->id]).'"><i class="fas fa-edit"></i>'.__('messages.edit').'</a></li>';
                    }

                    if (auth()->user()->can('purchase_requisition.delete')) {
                        $html .= '<li><a href="'.action([\App\Http\Controllers\PurchaseRequisitionController::class, 'destroy'], [$row->id]).'" class="delete-purchase-requisition"><i class="fas fa-trash"></i>'.__('messages.delete').'</a></li>';
                    }

                    $html .= '</ul></div>';

                    return $html;
                })
                ->removeColumn('id')
                ->editColumn('priority_level', function ($row) use ($priority_levels) {
                    $label = $priority_levels[$row->priority_level] ?? '';
                    $class = '';
                    if ($row->priority_level === 'rush') {
                        $class = 'label label-danger';
                    } elseif ($row->priority_level === 'sure_order') {
                        $class = 'label label-success';
                    }
                    if ($label === '') {
                        return '';
                    }
                    return '<span class="'.$class.'">'.e($label).'</span>';
                })
                ->editColumn('delivery_date', '@if(!empty($delivery_date)){{@format_datetime($delivery_date)}}@endif')
                ->editColumn('transaction_date', '{{@format_datetime($transaction_date)}}')
                ->editColumn('status', function ($row) {
                    $status = '';
                    $order_statuses = $this->purchaseRequisitionStatuses;
                    if (array_key_exists($row->status, $order_statuses)) {
                        $status = '<span class="label '.$order_statuses[$row->status]['class']
                            .'" >'.$order_statuses[$row->status]['label'].'</span>';
                    }

                    return $status;
                })
                ->setRowAttr([
                    'data-href' => function ($row) {
                        return  action([\App\Http\Controllers\PurchaseRequisitionController::class, 'show'], [$row->id]);
                    }, ])
                ->rawColumns(['status', 'action', 'priority_level'])
                ->make(true);
        }

        $business_locations = BusinessLocation::forDropdown($business_id);

        $purchaseRequisitionStatuses = [];
        foreach ($this->purchaseRequisitionStatuses as $key => $value) {
            $purchaseRequisitionStatuses[$key] = $value['label'];
        }

        return view('purchase_requisition.index')->with(compact('business_locations', 'purchaseRequisitionStatuses'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        if (! auth()->user()->can('purchase_requisition.create')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        $business_locations = Cache::remember("pr:blist:$business_id", 600, function() use ($business_id) {
            return BusinessLocation::forDropdown($business_id);
        });

        $categories = Cache::remember("pr:cat:$business_id", 600, function() use ($business_id) {
            return Category::forDropdown($business_id, 'product');
        });

        $priority_levels = $this->priorityLevels();

        $pr_statuses = $this->getAllowedPrStatuses();
        // If none are allowed, keep array empty; the save will reject or fallback safely.

        return view('purchase_requisition.create')->with(compact('business_locations', 'categories', 'pr_statuses', 'priority_levels'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if (! auth()->user()->can('purchase_requisition.create')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            $business_id = request()->session()->get('user.business_id');
            $priority_level = $this->sanitizePriorityLevel($request->input('priority_level'));

            $requested_status = $request->input('status');
            $requested_status = in_array($requested_status, array_keys($this->purchaseRequisitionStatuses)) ? $requested_status : null;
            $default_allowed = $this->getDefaultAllowedPrStatus();

            $transaction_data = [
                'business_id' => $business_id,
                'location_id' => $request->input('location_id'),
                'type' => 'purchase_requisition',
                // Use requested status if allowed; otherwise choose first allowed or fail later
                'status' => $requested_status ?? ($default_allowed ?? 'requested'),
                'created_by' => auth()->user()->id,
                'transaction_date' => \Carbon::now()->toDateTimeString(),
                'priority_level' => $priority_level,
            ];
            // Enforce permission for selecting any status on create
            if (! $this->canSetPrStatus($transaction_data['status'])) {
                if (! is_null($default_allowed)) {
                    $transaction_data['status'] = $default_allowed;
                } else {
                    abort(403, 'Unauthorized action. No allowed PR statuses to set.');
                }
            }

            $raw_delivery = $request->input('delivery_date');
            if (! empty($raw_delivery)) {
                // Accept HTML5 datetime-local (YYYY-MM-DDTHH:mm) or existing formatted text
                if (strpos($raw_delivery, 'T') !== false) {
                    $transaction_data['delivery_date'] = \Carbon\Carbon::parse($raw_delivery)->format('Y-m-d H:i:s');
                } else {
                    $transaction_data['delivery_date'] = $this->commonUtil->uf_date($raw_delivery, true);
                }
            } else {
                $transaction_data['delivery_date'] = null;
            }

            $purchase_lines = [];
            $incoming_lines = $request->input('purchases', []);
            if (!is_array($incoming_lines)) {
                $incoming_lines = [];
            }
            foreach ($incoming_lines as $purchase_line) {
                $quantity = isset($purchase_line['quantity']) ? $this->commonUtil->num_uf($purchase_line['quantity']) : 0;
                $secondary_unit_quantity = isset($purchase_line['secondary_unit_quantity']) ? $this->commonUtil->num_uf($purchase_line['secondary_unit_quantity']) : 0;
                $unit_cost = isset($purchase_line['unit_cost']) ? $this->commonUtil->num_uf($purchase_line['unit_cost']) : 0;

                if (! empty($quantity) || ! empty($secondary_unit_quantity)) {
                    $purchase_lines[] = [
                        'variation_id' => $purchase_line['variation_id'],
                        'product_id' => $purchase_line['product_id'],
                        'quantity' => $quantity,
                        'purchase_price' => $unit_cost,
                        'purchase_price_inc_tax' => $unit_cost,
                        'pp_without_discount' => $unit_cost,
                        'item_tax' => 0,
                        'secondary_unit_quantity' => $secondary_unit_quantity,
                    ];
                }
            }

            DB::beginTransaction();

            //Update reference count
            $ref_count = $this->commonUtil->setAndGetReferenceCount($transaction_data['type']);
            //Generate reference number
            if (empty($transaction_data['ref_no'])) {
                $transaction_data['ref_no'] = $this->commonUtil->generateReferenceNumber($transaction_data['type'], $ref_count);
            }

            // Optional remarks/notes
            if ($request->filled('additional_notes')) {
                $transaction_data['additional_notes'] = $request->input('additional_notes');
            }

            $purchase_requisition = Transaction::create($transaction_data);
            $purchase_requisition->purchase_lines()->createMany($purchase_lines);

            DB::commit();

            $output = ['success' => 1,
                'msg' => __('lang_v1.added_success'),
            ];
        } catch (\Exception $e) {
            DB::rollBack();

            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => 0,
                'msg' => __('messages.something_went_wrong'),
            ];
        }

        return redirect()->action([\App\Http\Controllers\PurchaseRequisitionController::class, 'index'])->with('status', $output);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        if (! auth()->user()->can('purchase_requisition.view_all') && ! auth()->user()->can('purchase_requisition.view_own')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');

        $query = Transaction::where('business_id', $business_id)
                        ->where('type', 'purchase_requisition')
                        ->where('id', $id)
                            ->with(
                                'purchase_lines',
                                'purchase_lines.product',
                                'purchase_lines.product.unit',
                                'purchase_lines.product.second_unit',
                                'purchase_lines.variations',
                                'purchase_lines.variations.product_variation',
                                'location',
                                'sales_person'
                            );

        $purchase = $query->firstOrFail();

        return view('purchase_requisition.show')
                ->with(compact('purchase'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        if (! auth()->user()->can('purchase_requisition.create')) {
            abort(403, 'Unauthorized action.');
        }
        $business_id = request()->session()->get('user.business_id');
        $purchase = Transaction::where('business_id', $business_id)
            ->where('type', 'purchase_requisition')
            ->with([
                'location',
                'purchase_lines',
                'purchase_lines.product',
                'purchase_lines.product.unit',
                'purchase_lines.product.second_unit',
                'purchase_lines.variations',
                'purchase_lines.variations.product_variation',
            ])
            ->findOrFail($id);
        if ($purchase->status === 'rejected') {
            return redirect()->action([\App\Http\Controllers\PurchaseRequisitionController::class, 'index'])
                ->with('status', ['success' => 0, 'msg' => __('messages.unauthorized') . ' - Cannot edit a rejected requisition.']);
        }
        $business_locations = Cache::remember("pr:blist:$business_id", 600, function() use ($business_id) {
            return BusinessLocation::forDropdown($business_id);
        });
        $categories = Cache::remember("pr:cat:$business_id", 600, function() use ($business_id) {
            return Category::forDropdown($business_id, 'product');
        });
        $priority_levels = $this->priorityLevels();
        $pr_statuses = $this->getAllowedPrStatuses();

        $txComments = TransactionComment::where('transaction_id', $purchase->id)
            ->orderBy('created_at', 'asc')
            ->get();

        // Preload current stock per variation at the purchase's location so the edit screen can show it
        $stocks_by_variation = [];
        if (!empty($purchase->location_id)) {
            foreach ($purchase->purchase_lines as $pl) {
                try {
                    $stocks_by_variation[$pl->variation_id] = (float) (VariationLocationDetails::where('variation_id', $pl->variation_id)
                        ->where('location_id', $purchase->location_id)
                        ->value('qty_available') ?? 0);
                } catch (\Throwable $e) {
                    $stocks_by_variation[$pl->variation_id] = 0;
                }
            }
        }

        return view('purchase_requisition.edit', compact('purchase','business_locations','categories','pr_statuses','txComments','stocks_by_variation','priority_levels'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        if (! auth()->user()->can('purchase_requisition.create')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $user_id = request()->session()->get('user.id');

        $purchase = Transaction::where('business_id', $business_id)
            ->where('type', 'purchase_requisition')
            ->findOrFail($id);
        if ($purchase->status === 'rejected') {
            return redirect()->action([\App\Http\Controllers\PurchaseRequisitionController::class, 'index'])
                ->with('status', ['success' => 0, 'msg' => __('messages.unauthorized') . ' - Cannot edit a rejected requisition.']);
        }

        $data = [];
        $priority_level = $this->sanitizePriorityLevel($request->input('priority_level'));
        if (!is_null($priority_level)) {
            $data['priority_level'] = $priority_level;
        } elseif ($request->has('priority_level')) {
            $data['priority_level'] = null;
        }
        if ($request->has('additional_notes')) {
            $data['additional_notes'] = $request->input('additional_notes');
        }
        if ($request->filled('delivery_date')) {
            $raw = $request->input('delivery_date');
            if (strpos($raw, 'T') !== false) {
                $data['delivery_date'] = \Carbon\Carbon::parse($raw)->format('Y-m-d H:i:s');
            } else {
                $data['delivery_date'] = $this->commonUtil->uf_date($raw, true);
            }
        } elseif ($request->has('delivery_date')) {
            $data['delivery_date'] = null;
        }
        if ($request->has('status') && in_array($request->input('status'), array_keys($this->purchaseRequisitionStatuses))) {
            $new_status = $request->input('status');
            if ($this->canSetPrStatus($new_status)) {
                $data['status'] = $new_status;
            } // else ignore unauthorized attempt to change status
        }
        $data['updated_by'] = $user_id;

        $purchase->fill($data);
        $purchase->save();

        // Update existing lines and add new ones if needed
        $incoming = (array) $request->input('purchases', []);
        $incoming_variation_ids = [];
        if (!empty($incoming)) {
            $existing = $purchase->purchase_lines->keyBy('variation_id');
            $to_create = [];
            foreach ($incoming as $pl) {
                $variation_id = isset($pl['variation_id']) ? (int) $pl['variation_id'] : null;
                $qty = isset($pl['quantity']) ? $this->commonUtil->num_uf($pl['quantity']) : 0;
                $sq = isset($pl['secondary_unit_quantity']) ? $this->commonUtil->num_uf($pl['secondary_unit_quantity']) : 0;
                $unit_cost = isset($pl['unit_cost']) ? $this->commonUtil->num_uf($pl['unit_cost']) : 0;
                if (!is_null($variation_id)) {
                    $incoming_variation_ids[] = $variation_id;
                }
                if ($variation_id && $existing->has($variation_id)) {
                    $line = $existing->get($variation_id);
                    $line->quantity = $qty;
                    $line->secondary_unit_quantity = $sq;
                    $line->purchase_price = $unit_cost;
                    $line->purchase_price_inc_tax = $unit_cost;
                    $line->pp_without_discount = $unit_cost;
                    $line->save();
                } else {
                    if (!empty($qty) || !empty($sq)) {
                        $to_create[] = [
                            'variation_id' => $variation_id,
                            'product_id' => $pl['product_id'] ?? null,
                            'quantity' => $qty,
                            'purchase_price' => $unit_cost,
                            'purchase_price_inc_tax' => $unit_cost,
                            'pp_without_discount' => $unit_cost,
                            'item_tax' => 0,
                            'secondary_unit_quantity' => $sq,
                        ];
                    }
                }
            }
            if (!empty($to_create)) {
                $purchase->purchase_lines()->createMany($to_create);
            }
        }

        // Handle deletions of existing lines
        $to_delete_variations = array_map('intval', (array) $request->input('purchases_delete', []));
        // If a variation is both re-added and marked for deletion in the same edit,
        // prefer keeping it (user removed then added back). Filter those out.
        if (!empty($incoming_variation_ids) && !empty($to_delete_variations)) {
            $to_delete_variations = array_values(array_diff($to_delete_variations, $incoming_variation_ids));
        }
        if (!empty($to_delete_variations)) {
            $lines_by_var = $purchase->purchase_lines()->whereIn('variation_id', $to_delete_variations)->get(['id','variation_id']);
            $line_ids = $lines_by_var->pluck('id')->all();
            if (!empty($line_ids)) {
                // Unlink any purchase lines referencing these PR lines
                \App\PurchaseLine::whereIn('purchase_requisition_line_id', $line_ids)->update(['purchase_requisition_line_id' => null]);
                $purchase->purchase_lines()->whereIn('id', $line_ids)->delete();
            }
        }

        return redirect()->action([\App\Http\Controllers\PurchaseRequisitionController::class, 'index'])
            ->with('status', ['success' => 1, 'msg' => __('lang_v1.updated_success')]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if (! auth()->user()->can('purchase_requisition.delete')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            if (request()->ajax()) {
                $business_id = request()->session()->get('user.business_id');

                $transaction = Transaction::where('business_id', $business_id)
                                ->where('type', 'purchase_requisition')
                                ->with(['purchase_lines'])
                                ->find($id);

                //unset purchase_order_line_id if set
                PurchaseLine::whereIn('purchase_requisition_line_id', $transaction->purchase_lines->pluck('id'))
                        ->update(['purchase_requisition_line_id' => null]);

                $transaction->delete();

                $output = ['success' => true,
                    'msg' => __('lang_v1.deleted_success'),
                ];
            }
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::emergency('File:'.$e->getFile().'Line:'.$e->getLine().'Message:'.$e->getMessage());

            $output = ['success' => false,
                'msg' => $e->getMessage(),
            ];
        }

        return $output;
    }

    public function getRequisitionProducts()
    {
        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');

            $query = VariationLocationDetails::join(
                'product_variations as pv',
                'variation_location_details.product_variation_id',
                '=',
                'pv.id'
            )
                    ->join(
                        'variations as v',
                        'variation_location_details.variation_id',
                        '=',
                        'v.id'
                    )
                    ->join(
                        'products as p',
                        'variation_location_details.product_id',
                        '=',
                        'p.id'
                    )
                    ->leftjoin(
                        'business_locations as l',
                        'variation_location_details.location_id',
                        '=',
                        'l.id'
                    )
                    ->leftjoin('units as u', 'p.unit_id', '=', 'u.id')
                    ->leftjoin('units as su', 'p.secondary_unit_id', '=', 'su.id')
                    ->where('p.business_id', $business_id)
                    ->where('p.enable_stock', 1)
                    ->where('p.is_inactive', 0)
                    ->whereNull('v.deleted_at')
                    ->whereNotNull('p.alert_quantity')
                    ->whereRaw('variation_location_details.qty_available <= p.alert_quantity');

            //Check for permitted locations of a user
            $permitted_locations = auth()->user()->permitted_locations();
            if ($permitted_locations != 'all') {
                $query->whereIn('variation_location_details.location_id', $permitted_locations);
            }

            if (! empty(request()->input('location_id'))) {
                $query->where('variation_location_details.location_id', request()->input('location_id'));
            }
            if (! empty(request()->input('brand_id'))) {
                $query->whereIn('p.brand_id', request()->input('brand_id'));
            }

            if (! empty(request()->input('category_id'))) {
                $query->whereIn('p.category_id', request()->input('category_id'));
            }

            $products = $query->select(
                'p.name as product',
                'p.type',
                'p.sku',
                'p.alert_quantity',
                'p.max_quantity',
                'pv.name as product_variation',
                'v.name as variation',
                'v.sub_sku',
                'l.name as location',
                'variation_location_details.qty_available as stock',
                'u.short_name as unit',
                'v.id as variation_id',
                'p.id as product_id',
                'u.allow_decimal',
                'su.short_name as second_unit',
                'su.allow_decimal as su_allow_decimal'

            )
            ->groupBy('v.id')
            ->get();

            return view('purchase_requisition.product_list')->with(compact('products'));
        }
    }

    /**
     * Return a single product row for PR by variation/location, ignoring alert thresholds.
     */
    public function getRequisitionProductRow()
    {
        if (!request()->ajax()) {
            abort(404);
        }

        $business_id = request()->session()->get('user.business_id');
        $variation_id = (int) request()->input('variation_id');
        $location_id = (int) request()->input('location_id');
        if ($variation_id <= 0) {
            return '';
        }

        // Allow adding even if there is no VLD row for this variation/location (treat stock as 0)
        $query = DB::table('variations as v')
            ->join('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->join('products as p', 'pv.product_id', '=', 'p.id')
            ->leftJoin('variation_location_details as variation_location_details', function ($join) use ($location_id) {
                $join->on('variation_location_details.variation_id', '=', 'v.id');
                if (!empty($location_id)) {
                    $join->where('variation_location_details.location_id', '=', $location_id);
                }
            })
            ->leftJoin('business_locations as l', 'variation_location_details.location_id', '=', 'l.id')
            ->leftJoin('units as u', 'p.unit_id', '=', 'u.id')
            ->leftJoin('units as su', 'p.secondary_unit_id', '=', 'su.id')
            ->where('p.business_id', $business_id)
            ->where('p.enable_stock', 1)
            ->where('p.is_inactive', 0)
            ->whereNull('v.deleted_at')
            ->where('v.id', $variation_id);

        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            // Permit null VLD (no stock row) to still return
            $query->where(function ($q) use ($permitted_locations) {
                $q->whereIn('variation_location_details.location_id', $permitted_locations)
                  ->orWhereNull('variation_location_details.location_id');
            });
        }

        $products = $query->select(
            'p.name as product',
            'p.type',
            'p.sku',
            'p.alert_quantity',
            'p.max_quantity',
            'pv.name as product_variation',
            'v.name as variation',
            'v.sub_sku',
            'l.name as location',
            DB::raw('COALESCE(variation_location_details.qty_available, 0) as stock'),
            'u.short_name as unit',
            'v.id as variation_id',
            'p.id as product_id',
            'u.allow_decimal',
            'su.short_name as second_unit',
            'su.allow_decimal as su_allow_decimal',
            'v.default_purchase_price'
        )->groupBy('v.id')->get();

        return view('purchase_requisition.product_list')->with(compact('products'));
    }

    public function getPurchaseRequisitions($location_id)
    {
        $business_id = request()->session()->get('user.business_id');

        $purchase_requisitions = Transaction::where('business_id', $business_id)
                        ->where('type', 'purchase_requisition')
                        ->where('approval_status', 'approved')
                        ->whereIn('status', ['partial', 'ordered'])
                        ->where('location_id', $location_id)
                        ->select('ref_no as text', 'id')
                        ->get();

        return $purchase_requisitions;
    }

    public function getPurchaseRequisitionLines($purchase_requisition_id)
    {
        $business_id = request()->session()->get('user.business_id');

        $purchase_requisition = Transaction::where('business_id', $business_id)
                        ->where('type', 'purchase_requisition')
                        ->where('approval_status', 'approved')
                        ->with(['purchase_lines', 'purchase_lines.variations',
                            'purchase_lines.product', 'purchase_lines.product.unit', 'purchase_lines.variations.product_variation', ])
                        ->findOrFail($purchase_requisition_id);

        $taxes = TaxRate::where('business_id', $business_id)
                            ->ExcludeForTaxGroup()
                            ->get();

        $sub_units_array = [];
        foreach ($purchase_requisition->purchase_lines as $pl) {
            $sub_units_array[$pl->id] = $this->transactionUtil->getSubUnits($business_id, $pl->product->unit->id, false, $pl->product_id);
        }
        $hide_tax = request()->session()->get('business.enable_inline_tax') == 1 ? '' : 'hide';
        $currency_details = $this->transactionUtil->purchaseCurrencyDetails($business_id);
        $row_count = request()->input('row_count');
        $is_purchase_order = true;
        $html = view('purchase_requisition.partials.purchase_requisition_lines')
                ->with(compact(
                    'purchase_requisition',
                    'taxes',
                    'hide_tax',
                    'currency_details',
                    'row_count',
                    'sub_units_array',
                    'is_purchase_order'
                ))->render();

        return [
            'html' => $html,
        ];
    }

    /**
     * Approve a purchase requisition.
     */
    public function approve($id)
    {
        if (! auth()->user()->can('purchase_requisition.approve')) {
            abort(403, 'Unauthorized action.');
        }

        $business_id = request()->session()->get('user.business_id');
        $user_id = request()->session()->get('user.id');

        try {
            $transaction = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_requisition')
                ->findOrFail($id);

            if ($transaction->approval_status !== 'approved') {
                $transaction->approval_status = 'approved';
                // Also mark the PR workflow status as approved so list + details reflect it
                $transaction->status = 'approved';
                $transaction->approved_by = $user_id;
                $transaction->approved_at = now();
                $transaction->updated_by = $user_id;
                $transaction->save();
            }

            // If a Purchase Order already exists for this PR, reuse it
            $existing_po = Transaction::where('business_id', $business_id)
                ->where('type', 'purchase_order')
                ->where('purchase_requisition_ids', 'like', '%"'.$transaction->id.'"%')
                ->first();

            if (! $existing_po) {
                // Create a minimal Purchase Order linked to this PR
                $po_data = [
                    'business_id' => $business_id,
                    'created_by' => $user_id,
                    'type' => 'purchase_order',
                    'status' => 'ordered',
                    'approval_status' => 'pending',
                    'location_id' => $transaction->location_id,
                    'transaction_date' => now(),
                    'delivery_date' => $transaction->delivery_date,
                    'total_before_tax' => 0,
                    'discount_type' => null,
                    'discount_amount' => 0,
                    'tax_id' => null,
                    'tax_amount' => 0,
                    'shipping_details' => null,
                    'shipping_charges' => 0,
                    'final_total' => 0,
                    'purchase_requisition_ids' => [$transaction->id],
                ];

                // Generate a reference number similar to controller@store
                $ref_count = $this->commonUtil->setAndGetReferenceCount('purchase_order', $business_id);
                $po_data['ref_no'] = $this->commonUtil->generateReferenceNumber('purchase_order', $ref_count, $business_id);

                $existing_po = Transaction::create($po_data);

                // Copy PR lines into the new PO as purchase lines
                $pr_full = Transaction::where('business_id', $business_id)
                    ->with(['purchase_lines'])
                    ->find($transaction->id);

                if ($pr_full && $pr_full->purchase_lines && $pr_full->purchase_lines->count()) {
                    $already_linked = $existing_po->purchase_lines()->pluck('purchase_requisition_line_id')->filter()->all();
                    $po_lines = [];
                    foreach ($pr_full->purchase_lines as $pl) {
                        if (!empty($already_linked) && in_array($pl->id, $already_linked)) {
                            continue;
                        }
                        $po_lines[] = [
                            'product_id' => $pl->product_id,
                            'variation_id' => $pl->variation_id,
                            'quantity' => $pl->quantity,
                            'pp_without_discount' => 0,
                            'discount_percent' => 0,
                            'purchase_price' => 0,
                            'purchase_price_inc_tax' => 0,
                            'item_tax' => 0,
                            'tax_id' => null,
                            'lot_number' => null,
                            'mfg_date' => null,
                            'exp_date' => null,
                            'sub_unit_id' => $pl->sub_unit_id ?? null,
                            'secondary_unit_quantity' => $pl->secondary_unit_quantity ?? 0,
                            'purchase_requisition_line_id' => $pl->id,
                        ];
                    }
                    if (!empty($po_lines)) {
                        $existing_po->purchase_lines()->createMany($po_lines);
                    }
                }
            }

            $output = ['success' => true, 'msg' => __('lang_v1.updated_success'), 'redirect' => action([\App\Http\Controllers\PurchaseOrderController::class, 'edit'], [$existing_po->id])];
        } catch (\Exception $e) {
            \Log::error($e);
            $output = ['success' => false, 'msg' => $e->getMessage()];
        }

        if (request()->ajax()) {
            return $output;
        }

        if (!empty($output['redirect'])) {
            return redirect($output['redirect'])->with('status', $output);
        }
        return back()->with('status', $output);
    }
}
